﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/general-advice")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class GeneralAdviceController : BaseController
    {
        /// <summary>
        /// The generalAdvice services.
        /// </summary>
        private readonly IGeneralAdviceService generalAdviceService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public GeneralAdviceController(IGeneralAdviceService generalAdvicesServices, IAuditLogService auditLogServices)
        {
            this.generalAdviceService = generalAdvicesServices;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch generalAdvices.
        /// </summary>
        /// <param name="model">
        /// The generalAdvice filter model.
        /// </param>
        /// <returns>
        /// The list of generalAdvices.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of generalAdvices.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<GeneralAdviceModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]GeneralAdviceFilterModel model)
        {
            model = (GeneralAdviceFilterModel)EmptyFilter.Handler(model);
            var generalAdvices = await this.generalAdviceService.FetchAsync(model);
            return this.Success(generalAdvices);
        }

        /// <summary>
        /// The add generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice added successfully.
        /// - 409 - generalAdvice already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody]GeneralAdviceModel model, [FromHeader] LocationHeader header)
        {
            model = (GeneralAdviceModel)EmptyFilter.Handler(model);
            var response = await this.generalAdviceService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given generalAdvice name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.GeneralAdvices,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.FullName} generalAdvice has been added.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("generalAdvice has been added successfully.");
        }

        /// <summary>
        /// The update generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice updated successfully.
        /// - 409 - generalAdvice already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody]GeneralAdviceModel model, [FromHeader] LocationHeader header)
        {
            model = (GeneralAdviceModel)EmptyFilter.Handler(model);
            var response = await this.generalAdviceService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given generalAdvice name has already been exists with us.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.GeneralAdvices,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow,
                LogDescription = $"{model.FullName} generalAdvice has been updated.",
                LocationId= Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("generalAdvice has been updated successfully.");
        }

        /// <summary>
        /// The delete generalAdvice.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - generalAdvice deleted successfully.
        /// - 409 - generalAdvice can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody]GeneralAdviceModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (GeneralAdviceModel)EmptyFilter.Handler(model);
                var response = await this.generalAdviceService.DeleteAsync(model.GeneralAdviceId);
                var responseProviderEncounter = await this.generalAdviceService.updateProviderEncounterAsync(model.GeneralAdviceId);
                if (response == 0)
                {
                    return this.ServerError();
                }

                var auditLogModel = new AuditLogModel
                {
                    AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.GeneralAdvices,
                    LogFrom = (int)AccountType.Administrator,
                    LogDate = DateTime.UtcNow,
                    LogDescription = $"{model.FullName} GeneralAdvice has been deleted.",
                    LocationId= Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLogModel);

                return this.Success("General Advice has been deleted successfully.");
            }
            catch (NpgsqlException exception)
            {
              return this.ServerError(exception.Message);
            }
        }
    }
}